---
c: Copyright (C) Daniel Stenberg, <daniel@haxx.se>, et al.
SPDX-License-Identifier: curl
Title: CURLINFO_COOKIELIST
Section: 3
Source: libcurl
See-also:
  - CURLOPT_COOKIELIST (3)
  - curl_easy_getinfo (3)
  - curl_easy_setopt (3)
Protocol:
  - HTTP
Added-in: 7.14.1
---

# NAME

CURLINFO_COOKIELIST - all known cookies

# SYNOPSIS

~~~c
#include <curl/curl.h>

CURLcode curl_easy_getinfo(CURL *handle, CURLINFO_COOKIELIST,
                           struct curl_slist **cookies);
~~~

# DESCRIPTION

Pass a pointer to a 'struct curl_slist *' to receive a linked-list of all
cookies curl knows (expired ones, too). Do not forget to call
curl_slist_free_all(3) on the list after it has been used. If there are no
cookies (cookies for the handle have not been enabled or simply none have been
received) the 'struct curl_slist *' is made a NULL pointer.

Cookies that were imported in the Set-Cookie format without a domain name are
not exported by this option.

# %PROTOCOLS%

# EXAMPLE

~~~c
int main(void)
{
  CURL *curl = curl_easy_init();
  if(curl) {
    CURLcode result;
    curl_easy_setopt(curl, CURLOPT_URL, "https://example.com");

    /* enable the cookie engine */
    curl_easy_setopt(curl, CURLOPT_COOKIEFILE, "");

    result = curl_easy_perform(curl);

    if(result == CURLE_OK) {
      /* extract all known cookies */
      struct curl_slist *cookies = NULL;
      result = curl_easy_getinfo(curl, CURLINFO_COOKIELIST, &cookies);
      if(!result && cookies) {
        /* a linked list of cookies in cookie file format */
        struct curl_slist *each = cookies;
        while(each) {
          printf("%s\n", each->data);
          each = each->next;
        }
        /* we must free these cookies when we are done */
        curl_slist_free_all(cookies);
      }
    }
    curl_easy_cleanup(curl);
  }
}
~~~

# %AVAILABILITY%

# RETURN VALUE

curl_easy_getinfo(3) returns a CURLcode indicating success or error.

CURLE_OK (0) means everything was OK, non-zero means an error occurred, see
libcurl-errors(3).
