// Handling for default and customized banners/prompts for the Miller REPL.

package repl

import (
	"fmt"
	"os"
	"runtime"

	"golang.org/x/term"

	"github.com/johnkerl/miller/v6/pkg/colorizer"
	"github.com/johnkerl/miller/v6/pkg/lib"
	"github.com/johnkerl/miller/v6/pkg/version"
)

const ENV_PRIMARY_PROMPT = "MLR_REPL_PS1"
const ENV_SECONDARY_PROMPT = "MLR_REPL_PS2"
const DEFAULT_PRIMARY_PROMPT = "[mlr] "
const DEFAULT_SECONDARY_PROMPT = "... "

func getInputIsTerminal() bool {
	if runtime.GOOS == "windows" && os.Getenv("MSYSTEM") != "" {
		// Sadly, term.IsTerminal doesn't work inside MSYS2 but does work
		// outside MSYS2. So in that case we simply return true so that the mlr
		// repl has a prompt.
		return true
	}
	return term.IsTerminal(int(os.Stdin.Fd()))
}

func getPrompt1() string {
	prompt1 := os.Getenv(ENV_PRIMARY_PROMPT)
	if prompt1 == "" {
		prompt1 = DEFAULT_PRIMARY_PROMPT
	}
	prompt1 = lib.UnbackslashStringLiteral(prompt1) // In case they want a "\n" in there
	return colorizer.MaybeColorizeREPLPS1(prompt1, true)
}

func getPrompt2() string {
	prompt2 := os.Getenv(ENV_SECONDARY_PROMPT)
	if prompt2 == "" {
		prompt2 = DEFAULT_SECONDARY_PROMPT
	}
	prompt2 = lib.UnbackslashStringLiteral(prompt2) // In case they want a "\n" in there
	return colorizer.MaybeColorizeREPLPS2(prompt2, true)
}

func (repl *Repl) printStartupBanner() {
	if repl.inputIsTerminal {
		fmt.Printf("Miller %s REPL for %s/%s/%s\n", version.STRING, runtime.GOOS, runtime.GOARCH, runtime.Version())
		fmt.Printf("Docs: %s\n", lib.DOC_URL)
		fmt.Printf("Type ':h' or ':help' for online help; ':q' or ':quit' to quit.\n")
	}
}

func (repl *Repl) printPrompt1() {
	if repl.inputIsTerminal && repl.showPrompts {
		fmt.Print(repl.prompt1)
	}
}

func (repl *Repl) printPrompt2() {
	if repl.inputIsTerminal && repl.showPrompts {
		fmt.Print(repl.prompt2)
	}
}
