#!/bin/sh
#
# This script logs out from all active iSCSI sessions, excluding those
# listed in /run/open-iscsi/shutdown-keep-sessions. That file is
# generated by umountiscsi.sh and determines which sessions should not
# be terminated.
#

ISCSIADM=/usr/sbin/iscsiadm
PIDFILE=/run/iscsid.pid

ISCSI_ROOT_KEEP_ALL_SESSIONS_AT_SHUTDOWN=0
if [ -f /etc/default/open-iscsi ]; then
	. /etc/default/open-iscsi
fi

if [ -f /etc/iscsi/iscsi.initramfs ] && [ $ISCSI_ROOT_KEEP_ALL_SESSIONS_AT_SHUTDOWN -eq 1 ]; then
	# Don't logout from any sessions if root is on initramfs and the
	# administrator wanted it that way.
	exit 0
fi

if [ ! -s $PIDFILE ] || ! kill -0 `sed -n 1p $PIDFILE` >/dev/null 2>/dev/null ; then
	# Don't logout from iSCSI sessions if daemon isn't running
	echo "iSCSI initiator daemon not running, not logging out from targets." >&2
	exit 1
fi

EXCLUDED_SESSIONS=""
if [ -f /run/open-iscsi/shutdown-keep-sessions ] ; then
	_EXCLUDED_SESSIONS=$(cat /run/open-iscsi/shutdown-keep-sessions)
	for s in ${_EXCLUDED_SESSIONS} ; do
		EXCLUDED_SESSIONS="${EXCLUDED_SESSIONS:+$EXCLUDED_SESSIONS }${s}"
	done
fi

# trivial case
if [ -z "$EXCLUDED_SESSIONS" ] ; then
	$ISCSIADM -m node --logoutall=all
	exit $?
fi

in_set() {
	eval _set=\$$1
	case "${_set}" in
		("$2"|*" $2"|"$2 "*|*" $2 "*) return 0 ;;
		(*)                           return 1 ;;
	esac
}

# go through all iSCSI sessions, but exclude those where we don't want
# to logout from
RC=0
for host_dir in /sys/devices/platform/host* ; do
	[ -d "$host_dir"/iscsi_host* ] || continue
	for session_dir in "$host_dir"/session* ; do
		if in_set EXCLUDED_SESSIONS "$session_dir" ; then
			continue
		fi
		$ISCSIADM -m session -r "$session_dir" --logout
		rc=$?
		if [ $rc -ne 0 ] ; then
			RC=1
		fi
	done
done

exit $RC
