import datetime

from django.db import models


class Place(models.Model):
    name = models.CharField(max_length=50)
    address = models.CharField(max_length=80)

    class Meta:
        ordering = ("name",)


class Restaurant(Place):
    serves_hot_dogs = models.BooleanField(default=False)
    serves_pizza = models.BooleanField(default=False)


class ItalianRestaurant(Restaurant):
    serves_gnocchi = models.BooleanField(default=False)


class ParkingLot(Place):
    # An explicit link to the parent (we can control the attribute name).
    parent = models.OneToOneField(
        Place, models.CASCADE, primary_key=True, parent_link=True
    )
    capacity = models.IntegerField()


class ParkingLot3(Place):
    # The parent_link connector need not be the pk on the model.
    primary_key = models.AutoField(primary_key=True)
    parent = models.OneToOneField(Place, models.CASCADE, parent_link=True)


class ParkingLot4(models.Model):
    # Test parent_link connector can be discovered in abstract classes.
    parent = models.OneToOneField(Place, models.CASCADE, parent_link=True)

    class Meta:
        abstract = True


class ParkingLot4A(ParkingLot4, Place):
    pass


class ParkingLot4B(Place, ParkingLot4):
    pass


class Supplier(models.Model):
    name = models.CharField(max_length=50)
    restaurant = models.ForeignKey(Restaurant, models.CASCADE)


class Wholesaler(Supplier):
    retailer = models.ForeignKey(
        Supplier, models.CASCADE, related_name="wholesale_supplier"
    )


class Parent(models.Model):
    created = models.DateTimeField(default=datetime.datetime.now)


class Child(Parent):
    name = models.CharField(max_length=10)


class SelfRefParent(models.Model):
    parent_data = models.IntegerField()
    self_data = models.ForeignKey("self", models.SET_NULL, null=True)


class SelfRefChild(SelfRefParent):
    child_data = models.IntegerField()


class Article(models.Model):
    headline = models.CharField(max_length=100)
    pub_date = models.DateTimeField()

    class Meta:
        ordering = ("-pub_date", "headline")


class ArticleWithAuthor(Article):
    author = models.CharField(max_length=100)


class M2MBase(models.Model):
    articles = models.ManyToManyField(Article)


class M2MChild(M2MBase):
    name = models.CharField(max_length=50)


class Evaluation(Article):
    quality = models.IntegerField()

    class Meta:
        abstract = True


class QualityControl(Evaluation):
    assignee = models.CharField(max_length=50)


class BaseM(models.Model):
    base_name = models.CharField(max_length=100)


class DerivedM(BaseM):
    customPK = models.IntegerField(primary_key=True)
    derived_name = models.CharField(max_length=100)


class AuditBase(models.Model):
    planned_date = models.DateField()

    class Meta:
        abstract = True
        verbose_name_plural = "Audits"


class CertificationAudit(AuditBase):
    class Meta(AuditBase.Meta):
        abstract = True


class InternalCertificationAudit(CertificationAudit):
    auditing_dept = models.CharField(max_length=20)


# Abstract classes don't get m2m tables autocreated.
class Person(models.Model):
    name = models.CharField(max_length=100)

    class Meta:
        ordering = ("name",)


class AbstractEvent(models.Model):
    name = models.CharField(max_length=100)
    attendees = models.ManyToManyField(Person, related_name="%(class)s_set")

    class Meta:
        abstract = True
        ordering = ("name",)


class BirthdayParty(AbstractEvent):
    pass


class BachelorParty(AbstractEvent):
    pass


class MessyBachelorParty(BachelorParty):
    pass


# Check concrete -> abstract -> concrete inheritance
class SearchableLocation(models.Model):
    keywords = models.CharField(max_length=255)


class Station(SearchableLocation):
    name = models.CharField(max_length=128)

    class Meta:
        abstract = True


class BusStation(Station):
    inbound = models.BooleanField(default=False)


class TrainStation(Station):
    zone = models.IntegerField()


class User(models.Model):
    username = models.CharField(max_length=30, unique=True)


class Profile(User):
    profile_id = models.AutoField(primary_key=True)
    extra = models.CharField(max_length=30, blank=True)


# Check concrete + concrete -> concrete -> concrete
class Politician(models.Model):
    politician_id = models.AutoField(primary_key=True)
    title = models.CharField(max_length=50)


class Congressman(Person, Politician):
    state = models.CharField(max_length=2)


class Senator(Congressman):
    pass
