use core:lang;
use core:asm;

// Implement support for Foo? instead of Maybe<Foo>
SrcName maybeType(SrcName t) on Compiler {
	RecPart maybe("Maybe");
	maybe.add(t);

	SrcName name(t.pos);
	name.add("core");
	name.add(maybe);
	name;
}

// Temporary solution for casting to maybe.
Expr asMaybe(Block block, Expr expr) on Compiler {
	Expr result = expr;
	Value eType = expr.result.type;

	Name name;
	name.add("core");
	name.add("Maybe", Value:[eType]);

	Named? t = block.scope.find(name);
	if (t as Type) {
		Name ctor("__ctor", Value:[ Value(t), eType ]);
		Named? c = Scope(t).find(ctor);
		if (c as Function) {
			result = CtorCall(expr.pos, block.scope, c, Actuals(expr));
		}
	}

	result;
}

/**
 * Null for use with maybe-types. Only usable in contexts where it can be auto-casted
 * to something usable.
 */
class NullExpr extends Expr {
	init(SrcPos pos) {
		init(pos);
	}

	ExprResult result() : override {
		// Standard result is void.
		Value();
	}

	// Always castable to a 'maybe' type.
	Int castPenalty(Value to) : override {
		if (to.isMaybe) {
			1;
		} else {
			-1;
		}
	}

	// Generate code.
	void code(CodeGen g, CodeResult r) : override {
		if (!r.needed())
			return;

		if (t = r.type.type) {
			Function? ctor = t.defaultCtor;
			if (ctor) {
				core:asm:Var var = r.location(g);

				// Call the constructor.
				g.l << lea(ptrA, var);
				Array<Operand> vals;
				vals << ptrA;
				ctor.localCall(g, vals, CodeResult(Value(), g.block), false);
				r.created(g);
			} else {
				throw SyntaxError(pos, "Failed to locate the default constructor of " + r.type.toS);
			}
		} else {
			throw SyntaxError(pos, "Can not be used when 'void' types are expected!");
		}
	}

	Str toS() : override {
		"null";
	}
}
