use ui;
use progvis:program;
use core:lang;

/**
 * Data for a thread in the program.
 */
class Thread extends Data {
	// Original thread.
	ProgThread src;

	// Stack frames.
	Frame[] frames;

	// Mirrorred data from the thread.
	Bool alive;
	Bool sleeping;
	Bool crashed;
	Bool afterReturn;
	Bool barriersAvailable;

	// Create.
	init(ProgThread thread) {
		init {
			src = thread;
		}
	}

	// Copy.
	init(Thread original) {
		init {
			src = original.src;
			frames = Frame[](original.frames);
			alive = original.alive;
			sleeping = original.sleeping;
			crashed = original.crashed;
			afterReturn = original.afterReturn;
		}
	}

	// Get current position.
	SrcPos? pos() {
		if (frames.any)
			return frames.last.pos;
		null;
	}

	// Get thread id.
	Nat id() { src.threadId; }

	// Traverse stack frames and find individual values.
	void update(World:Traversal t, unsafe:RawPtr object, Nat offset) : override {
		alive = src.alive;
		sleeping = src.sleeping;
		crashed = src.crashed;
		afterReturn = false;
		// In some cases, we can be executed before the thread has started properly.
		if (src.frames.any)
			afterReturn = src.frames.last.returned;
		barriersAvailable = src.barriersAvailable;

		if (!alive & !crashed) {
			// Remove frames, but not if we crashed (makes it easier to find the crash).
			frames.clear();
			return;
		}

		for (k, v in src.frames) {
			if (k >= frames.count)
				frames.push(Frame(v));
			if (frames[k].src !is v)
				frames[k] = Frame(v);
			frames[k].update(t, object, 0);
		}

		while (frames.count > src.frames.count)
			frames.pop();
	}

	// Traverse pointers.
	void traverse(World:Traversal t, unsafe:RawPtr object, Nat offset) : override {
		for (f in frames)
			f.traverse(t, object, 0);
	}

	// Title.
	Str title(World world) : override {
		"Thread " + src.threadId.toS;
	}

	// Type.
	Type? type() : override {
		null;
	}

	// Visit.
	void visit(DataVisitor v) : override {
		super:visit(v);
		for (f in frames)
			f.visit(v);
	}

	// Summary.
	protected void putSummary(StrBuf to, SummaryData data) : override {
		// Stack frames. We don't need a delimiter.
		for (f in frames) {
			f.summary(to, data);
		}
	}

	// Copy.
	protected Data copyObject() : override {
		Thread(this);
	}
	protected void copyData(Data->Data data) : override {
		for (Nat i = 0; i < frames.count; i++)
			if (x = frames[i].copy(data) as Frame)
				frames[i] = x;
	}

	// Stack frame.
	class Frame extends Composite {
		// Original stack frame.
		StackFrame src;

		// Source position for this stack frame.
		SrcPos pos;

		// Current offset in the function.
		Nat offset;

		// Create.
		init(StackFrame src) {
			init(null) {
				src = src;
			}
		}

		// Copy.
		init(Frame original) {
			init(original) {
				src = original.src;
				pos = original.pos;
			}
		}

		Str title(World world) : override {
			// This is safe: the same stack frame object does not modify its title.
			src.function;
		}

		void update(World:Traversal t, unsafe:RawPtr object, Nat offset) : override {
			pos = src.pos;
			offset = src.lastOffset;

			for (k, v in src.variables) {
				var ptr = unsafe:RawPtr(v.value);
				if (k >= parts.count) {
					parts << Part(v.name, 0, t.createStack(ptr), false);
				} else if (parts[k].name != v.name) {
					parts[k] = Part(v.name, 0, t.createStack(ptr), false);
				} else {
					var oldType = parts[k].value.type;
					var newType = ptr.type;
					if (oldType.empty | newType.empty) {
						// No need to replace if any of them said null.
					} else if (newType !is oldType) {
						parts[k].value = t.createStack(ptr);
					}
				}

				parts[k].value.update(t, ptr, 0);

				// Don't add the object if it is represented as a StackPointer object. The
				// StackPointer object treats the data as a pointer to the data, and not the value
				// itself. If we add that, all StackData objects will simply point to themselves.
				if (parts[k].value as StackPointer)
					;
				else
					t.addData(ptr, parts[k].value);
			}
			while (parts.count > src.variables.count)
				parts.pop();
		}

		void traverse(World:Traversal t, unsafe:RawPtr object, Nat offset) : override {
			for (k, v in src.variables) {
				var ptr = unsafe:RawPtr(v.value);
				parts[k].value.traverse(t, ptr, 0);
			}
		}

		protected Data copyObject() : override {
			Frame(this);
		}

		protected void putSummary(StrBuf to, SummaryData data) : override {
			// Put the location first:
			if (file = pos.file)
				to << file.title << ":" << pos.start << "-" << pos.end << "@" << offset << ".";
			super:putSummary(to, data);
		}
	}

}
