use core:io;
use core:lang;
use markdown;
use lang:bs:macro;

/**
 * A description of a theme for documentation generation.
 *
 * The theme specifies headers and footers for HTML outputs, how to generate table of contents,
 * navigation, and if any additional files are necessary.
 */
class Theme {
	// Called once at startup to initialize the theme with any additional parameters supplied. The
	// theme is expected to remove any keys that it utilizes. This allows the system to report an
	// error for any keys that have been misspelled.
	void options(Str->Str options) {}

	// Called once for each tree before generation of HTML begins. This allows the theme to generate
	// any auxiliary data structures that might be necessary during generation. For example, it
	// allows the theme to pre-compute a table of contents.
	void initialize(Tree tree) {}

	// Called once for each document to generate the HTML code for a node in the tree. The default
	// implementation simply calls `document.toHtml().toS()`. Themes might wish to modify the
	// document beforehand, or add headers and footers to the generated HTML.
	Str toHtml(Node node, Document document) {
		document.toHtml().toS();
	}

	// Called once to copy any additional files to the output directory.
	void copyFiles(Url outputRoot) {}
}

// Load a theme.
Theme loadTheme(Str theme) {
	if (theme.empty) {
		print("Warning: No theme selected, output will not be valid HTML.");
		return Theme();
	}

	var fn = findThemeFn(theme);
	return fn.call();
}

private Fn<Theme> findThemeFn(Str theme) {
	var name = parseSimpleName(theme);

	// Will look in "themes" first, then in the root.
	Scope scope(named{themes});
	if (found = scope.find(name) as Function) {
		if (ptr = found.pointer as Fn<Theme>) {
			return ptr;
		}
	}

	throw DocError(Url(), "Can not find the theme \"${theme}\".");
}
