use util:serialize;
use core:info;

/**
 * This class is the base class for all requests (sent from the client to the server). We rely on
 * the built-in serialization for sending and receiving them.
 */
class Request : serializable {
	// Called by the server to execute the request. Returns a suitable message, or "null" if the
	// client should be disconnected.
	Response? execute(ServerConn server, Database db) {
		error("Unknown message");
	}

	// Helper to create an error message.
	protected Response error(Str message) {
		ErrorResponse(message);
	}
}


/**
 * Authentication request. Expected to be the first message sent by a client.
 *
 * Returns either an AuthReply, AuthLoginReply, or an error.
 */
class AuthRequest : extends Request, serializable {
	// The client key.
	Str key;

	// Version of the client. The default value means that if the version field is missing from the
	// client, the server will detect it as too old.
	Version version = Version(0, 0, 0);

	// Create.
	init(Str key, Version version) {
		init {
			key = key;
			version = version;
		}
	}

	// NOTE: This is handled as a special case, so we don't override "execute" here.
}

/**
 * Log out from this client. No additional data needed.
 */
class LogoutRequest : extends Request, serializable {
	Response? execute(ServerConn server, Database db) : override {
		db.logout(server.clientId);
		LogoutResponse();
	}
}

/**
 * Change nickname.
 */
class ChangeNicknameRequest : extends Request, serializable {
	Str newName;

	init(Str newName) {
		init { newName = newName; }
	}

	Response? execute(ServerConn server, Database db) : override {
		db.changeName(server.userId, newName);
		AuthResponse(newName, server.isAdmin);
	}
}

/**
 * Ask for the leaderboard.
 */
class LeaderboardRequest : extends Request, serializable {
	Response? execute(ServerConn server, Database db) : override {
		ScorePair[] scores;
		for (user, score in db.allScores()) {
			scores << ScorePair(user, score);
		}

		scores.sort();

		// Pick the top 10, and the current user.
		Bool currentFound = false;
		Score[] result;
		Int place = 1;
		for (i, score in scores) {
			Bool add = (score.user == server.userId);
			currentFound |= add;
			add |= place < 11;

			if (currentFound & !add)
				break;

			if (add) {
				if (user = db.findUser(score.user)) {
					if (!user.admin) {
						result << Score(user.name, score.score, place++);
					}
				}
			}
		}

		LeaderboardResponse(result);
	}
}

// Pair for use when working with the scoreboard.
private value ScorePair {
	Int user;
	Int score;

	init(Int user, Int score) {
		init { user = user; score = score; }
	}

	Bool <(ScorePair o) {
		score > o.score;
	}
}


/**
 * Publish a new problem.
 */
class NewProblemRequest : extends Request, serializable {
	Str title;
	Code impl;
	Code test;
	Code refImpl;
	ProblemChecks checks;
	AcceptableErrors acceptable;

	// Is there a concurrency error in the combination main + impl currently?
	Error currentError;

	init(Str title, Code impl, Code test, Code refImpl, ProblemChecks checks, AcceptableErrors accept, Error error) {
		init {
			title = title;
			test = test;
			impl = impl;
			refImpl = refImpl;
			checks = checks;
			acceptable = accept;
			currentError = error;
		}
	}

	Response? execute(ServerConn server, Database db) : override {
		Int id = db.createProblem(server.userId, title, impl, test, refImpl, checks, acceptable, currentError);
		NewItemResponse(id);
	}
}


/**
 * Request for all problems.
 */
class ProblemListRequest : extends Request, serializable {
	init() {}

	Response? execute(ServerConn server, Database db) : override {
		ProblemListResponse(db.problemList(server.userId));
	}
}


/**
 * Request a problem.
 */
class ProblemRequest : extends Request, serializable {
	Int problemId;
	Int implId;
	Int testId;

	init(Int problemId) {
		init {
			problemId = problemId;
			implId = -1;
			testId = -1;
		}
	}

	init(Int problemId, Int implId, Int testId) {
		init {
			problemId = problemId;
			implId = implId;
			testId = testId;
		}
	}

	init(CodeTriplet triplet) {
		init {
			problemId = triplet.problem;
			implId = triplet.implementation;
			testId = triplet.test;
		}
	}

	Response? execute(ServerConn server, Database db) : override {
		ProblemResponse(db.problem(problemId, implId, testId));
	}
}


/**
 * Request a problem, but where only one of the improvements is given. The other part is inferred
 * from the user's current state.
 */
class ImprovedProblemRequest : extends Request, serializable {
	Int problemId;
	Int improvementId;
	ImprovementType type;

	init(Int problemId, Int impId, ImprovementType type) {
		init {
			problemId = problemId;
			improvementId = impId;
			type = type;
		}
	}

	Response? execute(ServerConn server, Database db) : override {
		ProblemResponse(db.improvedProblem(server.userId, problemId, improvementId, type));
	}
}


/**
 * Save a sequence of operations as a solution attempt.
 */
class NewAttemptRequest : extends Request, serializable {
	Int implCodeId;
	Int testCodeId;
	Str sequence;
	Str? error;

	init(Int implId, Int testId, Str sequence, Str? error) {
		init() {
			implCodeId = implId;
			testCodeId = testId;
			sequence = sequence;
			error = error;
		}
	}

	Response? execute(ServerConn server, Database db) : override {
		NewItemResponse(db.newAttempt(server.userId, implCodeId, testCodeId, sequence, error));
	}

}


/**
 * Save a new implementation for a user.
 */
class NewImplementation : extends Request, serializable {
	// ID of the problem.
	Int problemId;

	// New implementation.
	Code newImpl;

	// Id of the test used, or -1 if the original test case was used.
	Int testId;

	// What error (if any) was found?
	Error error;

	// Create.
	init(Int problemId, Code newImpl, Int testId, Error error) {
		init() {
			problemId = problemId;
			newImpl = newImpl;
			testId = testId;
			error = error;
		}
	}

	Response? execute(ServerConn server, Database db) : override {
		NewItemResponse(db.newImplementation(server.userId, problemId, newImpl, testId, error));
	}
}


/**
 * Save a new test for a user.
 */
class NewTest : extends Request, serializable {
	// ID of the problem.
	Int problemId;

	// New test.
	Code newTest;

	// Id of the implementation used, or -1 if the original implementation was used.
	Int implId;

	// What error (if any) was found? NULL means no error was found.
	Error error;

	// Create.
	init(Int problemId, Code newTest, Int implId, Error error) {
		init() {
			problemId = problemId;
			newTest = newTest;
			implId = implId;
			error = error;
		}
	}

	Response? execute(ServerConn server, Database db) : override {
		NewItemResponse(db.newTest(server.userId, problemId, newTest, implId, error));
	}
}


/**
 * Request information about the current state of a problem.
 */
class CurrentStateRequest : extends Request, serializable {
	// ID of the problem.
	Int problemId;

	init(Int id) {
		init { problemId = id; }
	}

	Response? execute(ServerConn server, Database db) : override {
		db.currentState(server.userId, problemId);
	}
}


/**
 * Request information about tests that are available to explore.
 */
class ImprovementsRequest : extends Request, serializable {
	// ID of the problem.
	Int problemId;

	// What to request (i.e. if type is 'implementation' we get a list of implementations back, used
	// to create an improved test).
	ImprovementType type;

	init(Int id, ImprovementType type) {
		init {
			problemId = id;
			type = type;
		}
	}

	Response? execute(ServerConn server, Database db) : override {
		var result = if (type == ImprovementType:test) {
			db.testImprovements(server.userId, problemId);
		} else {
			db.implImprovements(server.userId, problemId);
		};
		ImprovementsResponse(problemId, result);
	}
}


/**
 * Request the code in a particular improvement.
 */
class ImprovementCodeRequest : extends Request, serializable {
	// Id of the improvement.
	Int improvementId;

	// Test or implementation?
	ImprovementType type;

	init(Int id, ImprovementType type) {
		init {
			improvementId = id;
			type = type;
		}
	}

	Response? execute(ServerConn server, Database db) : override {
		var result = if (type == ImprovementType:test) {
			db.testCode(improvementId);
		} else {
			db.implCode(improvementId);
		};
		CodeResponse(result);
	}
}


/**
 * Ask for all error combinations that are currently missing. Useful to remedy situations where
 * `implImprovements` or `testImprovements` return "Not checked yet".
 */
class MissingChecksRequest : extends Request, serializable {
	// ID of the problem.
	Int problemId;

	// ID of the improvement.
	Int impId;

	// Is 'impId' referring to an implementation or a test?
	ImprovementType type;

	init(Int problemId, Int impId, ImprovementType type) {
		init {
			problemId = problemId;
			impId = impId;
			type = type;
		}
	}

	Response? execute(ServerConn server, Database db) : override {
		if (type == ImprovementType:test) {
			return MissingChecksResponse(db.testMissing(server.userId, problemId, impId));
		} else {
			return MissingChecksResponse(db.implMissing(server.userId, problemId, impId));
		}
	}
}


/**
 * Set current error state of a problem combination.
 */
class StoreCheckResult : extends Request, serializable {
	Int problemId;
	Int implId;
	Int testId;
	Error error;

	init(Int problemId, Int implId, Int testId, Error error) {
		init {
			problemId = problemId;
			implId = implId;
			testId = testId;
			error = error;
		}
	}

	Response? execute(ServerConn server, Database db) : override {
		db.addError(problemId, implId, testId, error);
		Response();
	}
}
