use lang:bs;
use core:lang;
use core:asm; // Note: also contains a Block class!

/**
 * An expression representing creating a template syntax tree.
 */
class TemplateExpr extends ExprBlock {
	init(SrcPos pos, Block block, Expr env, SExpr create) {
		init(create.pos, block);
		var atoms = create.allChildren(named{PatternAtom});
		var types = create.allChildren(named{PatternType});
		var arrays = findArrays(create);

		Type? tag;

		// PatternBlock exprs(<env>, <tag>, atoms.count, types.count, actuals.count);
		Actuals params;
		params.add(env.expectCastTo(named{Block}, block.scope));

		if (types.any) {
			PatternTag t(pos);
			params.add(ReferNamed(pos, t));
			tag = t;
		} else {
			params.add(NullExpr(pos));
		}

		params.add(NumLiteral(pos, atoms.count.long));
		params.add(NumLiteral(pos, types.count.long));
		params.add(NumLiteral(pos, arrays.count.long));

		Var exprVars(this, Value(named{PatternBlock}), SStr("block"), params);
		add(exprVars);
		LocalVarAccess exprs(pos, exprVars.var);

		for (id, node in atoms) {
			if (node as PatternAtom) {
				addExpr(block, exprs, node.pattern, "pushExpr", "Expr");
				node.pattern = PatternExprSrc(id);
			} else {
				throw SyntaxError(pos, "Internal error!");
			}
		}

		for (id, node in types) {
			if (node as PatternType) {
				addExpr(block, exprs, node.pattern, "pushType", "Named");
				if (tag)
					node.pattern = PatternTypeSrc(tag, id);
			} else {
				throw SyntaxError(pos, "Internal error!");
			}
		}

		for (node in arrays) {
			addExpr(block, exprs, node, "pushArray", "Array<Expr>");
		}

		// Call transform on the stored syntax tree.
		ReferSExpr src(create);
		add(namedExpr(this, SStr("transform"), exprs, Actuals(src)));

		// Return 'exprs'.
		add(exprs);
	}

	void code(CodeGen state, CodeResult r) {
		super:code(state, r);
	}

private:
	Array<SPatternExpr> findArrays(SExpr create) {
		SPatternExpr[] result;
		for (x in create.allChildren(named{PatternActual})) {
			if (x as PatternActual) {
				result << x.pattern;
				x.pattern = PatternExprSrc(result.count - 1);
			}
		}
		for (x in create.allChildren(named{PatternArray})) {
			if (x as PatternArray) {
				result << x.pattern;
				x.pattern = PatternExprSrc(result.count - 1);
			}
		}
		for (x in create.allChildren(named{PatternBlockItem})) {
			if (x as PatternBlockItem) {
				result << x.pattern;
				x.pattern = PatternExprSrc(result.count - 1);
			}
		}
		result;
	}

	void addExpr(Block parent, LocalVarAccess exprs, SPatternExpr node, Str member, Str expected) {
		if (var = node as PatternVar) {
			addExpr(member, expected, exprs, namedExpr(parent, var.name.transform, Actuals()));
		} else if (expr = node as PatternExpr) {
			addExpr(member, expected, exprs, expr.expr.transform(parent));
		} else {
			throw SyntaxError(pos, "Unexpected subtype of SPatternExpr found. Can not continue.");
		}
	}

	void addExpr(Str member, Str expected, LocalVarAccess exprs, Expr insert) {
		var insertExpr = namedExpr(this, SStr(member), exprs, Actuals(insert));
		if (insertExpr as UnresolvedName) {
			throw SyntaxError(insert.pos, "The expression ${insert.toS} in the pattern does not evaluate to ${expected} as needed from its context.");
		}
		add(insertExpr);
	}
}


/**
 * Unique tag for a pattern block. We put this as a parameter of the first part of our name lookup
 * to be able to distinguish between different pattern blocks during name lookup. Never intended to
 * be instantiated.
 */
package class PatternTag extends Type {
	init(SrcPos pos) {
		init("<pattern tag>", TypeFlags:typeValue) {}
	}
}


/**
 * Block containing expressions to be inserted into a pattern.
 *
 * Attempts to appear invisible to the user. Variables added here are actually added to the parent scope.
 */
class PatternBlock extends Block {
	// Expressions to be inserted later.
	Array<Expr> exprs;

	// Types to be inserted later.
	Array<Named> types;

	// Actuals to be inserted later.
	Array<Array<Expr>> actuals;

	// Pattern tag type, if we need one.
	private Type? tag;

	// Parent block.
	private Block parent;

	// Contained expr.
	private Expr data;

	init(Block parent, Type? tag, Nat atomCount, Nat typesCount, Nat actualsCount) {
		init(parent.pos, parent) {
			parent = parent;
			tag = tag;
			// Create a dummy object for now.
			data(parent.pos);
		}

		exprs.reserve(atomCount);
		types.reserve(typesCount);
		actuals.reserve(actualsCount);

		// Create a custom lookup for this object:
		if (parent = lookup.parent) {
			lookup = PatternLookup(this, parent);
			scope = scope.child(lookup);
		}
	}

	// Add an expression.
	void pushExpr(Expr expr) {
		exprs.push(expr);
	}

	// Add a type.
	void pushType(Named type) {
		types.push(type);
	}

	// Add a set of actual parameters.
	void pushArray(Array<Expr> expr) {
		actuals.push(expr);
	}

	// Transform an SExpr and contain the new value.
	void transform(SExpr src) {
		data = src.transform(this);
	}

	// Do not add variables here. Add them to the parent scope instead!
	void add(LocalVar v) {
		parent.add(v);
	}

	// Result of this block.
	ExprResult result() {
		data.result();
	}

	// Generate code for this block.
	void blockCode(CodeGen state, CodeResult to) {
		data.code(state, to);
	}

	// To string.
	void toS(StrBuf to) {
		to << data;
	}

	// Find a named that was captured.
	Named? findNamed(SimplePart part) {
		unless (part.params.count == 1)
			return null;
		unless (part.params[0].type is tag)
			return null;

		// Now we know that it is for us!
		// Extract the ID:
		var lastSpace = part.name.findLast(' ');
		Str number = part.name.cut(lastSpace + 1);

		unless (index = number.nat)
			return null;

		if (index >= types.count)
			return null;

		return types[index];
	}
}


/**
 * Custom block lookup for pattern blocks.
 */
package class PatternLookup extends BlockLookup {
	// Create.
	init(PatternBlock block, NameLookup parent) {
		init(block, parent) {}
	}

	// Override the code to find things here.
	Named? find(SimplePart part, Scope source) : override {
		if (found = super:find(part, source))
			return found;

		if (block as PatternBlock)
			return block.findNamed(part);

		null;
	}
}


/**
 * Subclass to the SPatternExpr used to tell the transformation where to fetch the proper
 * expressions from.
 */
class PatternExprSrc extends SPatternExpr {
	// The index we're concerned of.
	private Nat id;

	// Create.
	init(Nat id) {
		init() {
			id = id;
		}
	}

	// Get an expression.
	Expr expr(Block block) {
		if (r = findPattern(block)) {
			r.exprs[id];
		} else {
			throw SyntaxError(block.pos, "Can not use \${} syntax outside of pattern blocks.");
		}
	}

	// Get an array.
	Array<Expr> array(Block block) {
		if (r = findPattern(block)) {
			r.actuals[id];
		} else {
			throw SyntaxError(block.pos, "Can not use @{} syntax outside of pattern blocks.");
		}
	}

	// Dummy transformation function.
	void transform() {}

	// To string.
	void toS(StrBuf to) {
		to << "Captured expression " << id;
	}

	private Maybe<PatternBlock> findPattern(Block src) {
		Block now = src;
		while (true) {
			if (now as PatternBlock)
				return now;

			if (l = now.lookup.parent as BlockLookup) {
				now = l.block;
			} else {
				return null;
			}
		}

		null;
	}
}


/**
 * Subclass to the SPatternExpr used to tell the transformation what name to emit in cases of type
 * names.
 */
class PatternTypeSrc extends SPatternExpr {
	// Tag we are going to use.
	private Type tag;

	// Index we wish to fetch.
	private Nat id;

	// Create.
	init(Type tag, Nat id) {
		init { tag = tag; id = id; }
	}

	// To string.
	void toS(StrBuf to) {
		to << "Captured expression " << id;
	}

	// Create a name.
	SrcName name(SrcPos pos) {
		SrcName name(pos);
		name.add("@pattern type " + id.toS, Value[](1, tag));
		return name;
	}
}


// Insert a previously generated Expr here.
Expr insertExpr(SrcPos pos, Block block, SPatternExpr pattern) on Compiler {
	if (pattern as PatternExprSrc) {
		pattern.expr(block);
	} else {
		throw SyntaxError(pos, "Can not use \${} syntax outside of pattern blocks.");
		// When we have exceptions, we can ignore this.
		Expr(pos);
	}
}

// Insert a type.
SrcName insertType(SrcPos pos, SPatternExpr pattern) on Compiler {
	if (pattern as PatternTypeSrc) {
		pattern.name(pos);
	} else {
		throw SyntaxError(pos, "Can not use \${} syntax outside of pattern blocks.");
	}
}

// Insert a previously generated Array<Expr> here.
void insertActuals(SrcPos pos, Actuals actuals, Block block, SPatternExpr pattern) on Compiler {
	if (pattern as PatternExprSrc) {
		for (x in pattern.array(block)) {
			actuals.add(x);
		}
	} else {
		throw SyntaxError(pos, "Can not use @{} syntax outside of pattern blocks.");
	}
}

// Insert a previously generated Array<Expr> here.
void insertExprs(SrcPos pos, Array<Expr> into, Block block, SPatternExpr pattern) on Compiler {
	if (pattern as PatternExprSrc) {
		for (x in pattern.array(block)) {
			into.push(x);
		}
	} else {
		throw SyntaxError(pos, "Can not use @{} syntax outside of pattern blocks.");
	}
}

void insertExprs(SrcPos pos, ExprBlock block, SPatternExpr pattern) on Compiler {
	if (pattern as PatternExprSrc) {
		for (x in pattern.array(block)) {
			block.add(x);
		}
	} else {
		throw SyntaxError(pos, "Can not use @{} syntax outside of pattern blocks.");
	}
}

/**
 * Refer to a SExpr object.
 */
class ReferSExpr extends Expr {
	// The expression referred to.
	private SExpr expr;

	init(SExpr expr) {
		init(expr.pos) { expr = expr; }
	}

	ExprResult result() {
		ExprResult(Value(named{lang:bs:SExpr}));
	}

	void code(CodeGen state, CodeResult result) {
		if (result.needed) {
			var v = result.location(state);
			state.l << mov(v, objPtr(expr));
			result.created(state);
		}
	}

	Str toS() {
		"<captured syntax tree>";
	}
}
